const ranges = [
    { minRange: 1, maxRange: 40, tag: "T , Foil" },
    { minRange: 1, maxRange: 101, tag: "C , Foil" },
    { minRange: 1, maxRange: 101, tag: "C , Foil" },
    { minRange: 1, maxRange: 101, tag: "C , Foil" },
    { minRange: 1, maxRange: 101, tag: "C , Foil" },
    { minRange: 1, maxRange: 80, tag: "U , Foil" },
    { minRange: 1, maxRange: 80, tag: "U , Foil" },
    { minRange: 1, maxRange: 80, tag: "U , Foil" },
    { minRange: 1, maxRange: 15, tag: "L , Foil" },
    { minRange: 1, maxRange: 18, tag: "U WOT Enchanting , Normal" },
    { minRange: 1, maxRange: 18, tag: "U WOT Enchanting , Foil" },
    { minRange: 1, maxRange: 100, tag: "Slot12" },
    { minRange: 1, maxRange: 60, tag: "R , Foil" },
    { minRange: 1, maxRange: 20, tag: "M , Foil" },
    { minRange: 1, maxRange: 100, tag: "Slot13" },
    { minRange: 1, maxRange: 52, tag: "Extended Art , Normal" },
    { minRange: 1, maxRange: 28, tag: "WOC Extended Art , Normal" },
    { minRange: 1, maxRange: 28, tag: "WOC Extended Art , Foil" },
    { minRange: 1, maxRange: 100, tag: "Slot14" },
    { minRange: 1, maxRange: 13, tag: "R Show , Normal" },
    { minRange: 1, maxRange: 7, tag: "M Show , Normal" },
    { minRange: 1, maxRange: 5, tag: "R Borderless , Normal" },
    { minRange: 1, maxRange: 6, tag: "M Borderless , Normal" },
    { minRange: 1, maxRange: 100, tag: "Slot15" },
    { minRange: 1, maxRange: 30, tag: "R WOT Enchanting , Normal" },
    { minRange: 1, maxRange: 15, tag: "M WOT Enchanting , Normal" },
    { minRange: 1, maxRange: 5, tag: "R WOT Borderless , Normal" },
    { minRange: 1, maxRange: 15, tag: "M WOT Borderless , Normal" },
    { minRange: 1, maxRange: 100, tag: "Slot16" },
    { minRange: 1, maxRange: 47, tag: "Extended Art , Foil" },
    { minRange: 1, maxRange: 13, tag: "R Show , Foil" },
    { minRange: 1, maxRange: 7, tag: "M Show , Foil" },
    { minRange: 1, maxRange: 5, tag: "R Borderless , Foil" },
    { minRange: 1, maxRange: 6, tag: "M Borderless , Foil" },
    { minRange: 1, maxRange: 30, tag: "R WOT Enchanting , Foil" },
    { minRange: 1, maxRange: 15, tag: "M WOT Enchanting , Foil" },
    { minRange: 1, maxRange: 5, tag: "R WOT Borderless , Foil" },
    { minRange: 1, maxRange: 15, tag: "M WOT Borderless , Foil" },
    { minRange: 1, maxRange: 5, tag: "R WOT Confetti , Foil" },
    { minRange: 1, maxRange: 15, tag: "M WOT Confetti , Foil" },
];

function incrementNonce(nonce, offset) {
  let seconds = parseInt(nonce.slice(0, 2), 10);
  let minutes = parseInt(nonce.slice(2, 4), 10);
  let hours = parseInt(nonce.slice(4, 6), 10);
  let day = parseInt(nonce.slice(6, 8), 10);
  let month = parseInt(nonce.slice(8, 10), 10);
  let year = parseInt(nonce.slice(10, 14), 10);

  seconds += offset;
  while (seconds > 59) {
    seconds -= 60;
    minutes += 1;
    if (minutes > 59) {
      minutes = 0;
      hours += 1;
      if (hours > 23) {
        hours = 0;
        day += 1;
        const daysInMonth = [31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31];
        if (day > daysInMonth[month - 1]) {
          day = 1;
          month += 1;
          if (month > 12) {
            month = 1;
            year += 1;
          }
        }
      }
    }
  }

  return (
    seconds.toString().padStart(2, '0') +
    minutes.toString().padStart(2, '0') +
    hours.toString().padStart(2, '0') +
    day.toString().padStart(2, '0') +
    month.toString().padStart(2, '0') +
    year.toString().padStart(4, '0')
  );
}

async function generateHmacSha512(key, message) {
  const keyBuffer = new TextEncoder().encode(key);
  const msgBuffer = new TextEncoder().encode(message);
  const cryptoKey = await crypto.subtle.importKey(
    "raw",
    keyBuffer,
    { name: "HMAC", hash: "SHA-512" },
    false,
    ["sign"]
  );
  const signature = await crypto.subtle.sign("HMAC", cryptoKey, msgBuffer);
  return new Uint8Array(signature);
}

async function generateRandomNumbers(ranges, serverSeed, clientSeed, nonce) {
  const randomNumbers = [];
  const numbersPerDigest = 16;
  const digestsNeeded = Math.ceil(ranges.length / numbersPerDigest);

  for (let digestIndex = 0; digestIndex < digestsNeeded; digestIndex++) {
    const message = `${clientSeed}:${nonce}:${digestIndex}`;
    const digest = await generateHmacSha512(serverSeed, message);
    let offset = 0;

    for (let i = 0; i < numbersPerDigest && randomNumbers.length < ranges.length; i++) {
      const range = ranges[randomNumbers.length];
      const bytes = digest.slice(offset, offset + 4);
      const view = new DataView(bytes.buffer, bytes.byteOffset, 4);
      const num = view.getUint32(0, false); // Big-endian
      const rand = Math.floor((num / 0x100000000) * (range.maxRange - range.minRange + 1)) + range.minRange;
      randomNumbers.push(rand);
      offset += 4;
    }
  }

  return randomNumbers;
}

async function mapTaggedResults(randomNumbers, ranges, serverSeed, clientSeed, nonce) {
  const seenValues = {};
  const taggedResults = [];

  for (let index = 0; index < randomNumbers.length; index++) {
    if (!ranges[index]) {
      console.error(`No range found for index ${index}`);
      continue;
    }

    const { tag = "UNKNOWN", minRange, maxRange } = ranges[index];
    let finalNumber = randomNumbers[index];
    let nonceOffset = 0;

    if (!seenValues[tag]) {
      seenValues[tag] = new Set();
    }

    while (seenValues[tag].has(finalNumber.toString())) {
      nonceOffset++;
      const newNonce = incrementNonce(nonce, nonceOffset);
      const message = `${clientSeed}:${newNonce}`;
      const digest = await generateHmacSha512(serverSeed, message);
      const bytes = digest.slice(0, 4);
      const view = new DataView(bytes.buffer, bytes.byteOffset, 4);
      const newNum = view.getUint32(0, false); // Big-endian
      finalNumber = Math.floor((newNum / 0x100000000) * (maxRange - minRange + 1)) + minRange;
    }

    seenValues[tag].add(finalNumber.toString());
    taggedResults.push({ value: finalNumber.toString(), tag });
  }

  return taggedResults;
}

async function filterResults(taggedResults, ranges, serverSeed, clientSeed, nonce) {
    let filteredResults = [];
    const usedIndexes = new Set();

    const findAndConsume = (tag) => {
        const item = taggedResults.find((item, i) => item.tag === tag && !usedIndexes.has(i));
        if (item) usedIndexes.add(taggedResults.indexOf(item));
        return item;
    };

    const add = (item) => { if (item) filteredResults.push(item); };

    let tFrameFoil = taggedResults.filter((item, i) => item.tag === "T , Foil" && !usedIndexes.has(i));
    tFrameFoil.slice(0, 1).forEach(item => {
        usedIndexes.add(taggedResults.indexOf(item));
        filteredResults.push(item);
    });

    let cFoil = taggedResults.filter((item, i) => item.tag === "C , Foil" && !usedIndexes.has(i));
    cFoil.slice(0, 4).forEach(item => {
        usedIndexes.add(taggedResults.indexOf(item));
        filteredResults.push(item);
    });

    let uFoil = taggedResults.filter((item, i) => item.tag === "U , Foil" && !usedIndexes.has(i));
    uFoil.slice(0, 3).forEach(item => {
        usedIndexes.add(taggedResults.indexOf(item));
        filteredResults.push(item);
    });

    let lFoil = taggedResults.filter((item, i) => item.tag === "L , Foil" && !usedIndexes.has(i));
    lFoil.slice(0, 1).forEach(item => {
        usedIndexes.add(taggedResults.indexOf(item));
        filteredResults.push(item);
    });

    let wNormal = taggedResults.filter((item, i) => item.tag === "U WOT Enchanting , Normal" && !usedIndexes.has(i));
    wNormal.slice(0, 1).forEach(item => {
        usedIndexes.add(taggedResults.indexOf(item));
        filteredResults.push(item);
    });

    let wFoil = taggedResults.filter((item, i) => item.tag === "U WOT Enchanting , Foil" && !usedIndexes.has(i));
    wFoil.slice(0, 1).forEach(item => {
        usedIndexes.add(taggedResults.indexOf(item));
        filteredResults.push(item);
    });

    const slot12 = taggedResults.find(item => item.tag === "Slot12");
    if (slot12) {
        const num = parseInt(slot12.value, 10);
        if (num <= 86) add(findAndConsume("R , Foil"));
        else add(findAndConsume("M , Foil"));
    }

    const slot13 = taggedResults.find(item => item.tag === "Slot13");
    if (slot13) {
        const num = parseInt(slot13.value, 10);
        if (num <= 71) add(findAndConsume("Extended Art , Normal"));
        else if (num <= 89) add(findAndConsume("WOC Extended Art , Normal"));
        else add(findAndConsume("WOC Extended Art , Foil"));
    }

    const slot14 = taggedResults.find(item => item.tag === "Slot14");
    if (slot14) {
        const num = parseInt(slot14.value, 10);
        if (num <= 54) add(findAndConsume("R Show , Normal"));
        else if (num <= 68) add(findAndConsume("M Show , Normal"));
        else if (num <= 89) add(findAndConsume("R Borderless , Normal"));
        else add(findAndConsume("M Borderless , Normal"));
    }

    const slot15 = taggedResults.find(item => item.tag === "Slot15");
    if (slot15) {
        const num = parseInt(slot15.value, 10);
        if (num <= 73) add(findAndConsume("R WOT Enchanting , Normal"));
        else if (num <= 83) add(findAndConsume("M WOT Enchanting , Normal"));
        else if (num <= 94) add(findAndConsume("R WOT Borderless , Normal"));
        else add(findAndConsume("M WOT Borderless , Normal"));
    }

    const slot16 = taggedResults.find(item => item.tag === "Slot16");
    if (slot16) {
        const num = parseInt(slot16.value, 10);
        if (num <= 43) add(findAndConsume("Extended Art , Foil"));
        else if (num <= 54) add(findAndConsume("R Show , Foil"));
        else if (num <= 60) add(findAndConsume("M Show , Foil"));
        else if (num <= 64) add(findAndConsume("R Borderless , Foil"));
        else if (num <= 67) add(findAndConsume("M Borderless , Foil"));
        else if (num <= 91) add(findAndConsume("R WOT Enchanting , Foil"));
        else if (num <= 94) add(findAndConsume("M WOT Enchanting , Foil"));
        else if (num <= 96) add(findAndConsume("R WOT Borderless , Foil"));
        else if (num <= 97) add(findAndConsume("M WOT Borderless , Foil"));
        else if (num <= 98) add(findAndConsume("R WOT Confetti , Foil"));
        else add(findAndConsume("M WOT Confetti , Foil"));
    }

    return filteredResults;
}

async function setData(serverSeed, clientSeed, nonce) {
  const randomNumbers = await generateRandomNumbers(ranges, serverSeed, clientSeed, nonce);
  const taggedResults = await mapTaggedResults(randomNumbers, ranges, serverSeed, clientSeed, nonce);
  const filteredResults = await filterResults(taggedResults, ranges, serverSeed, clientSeed, nonce);

  return {
    success: true,
    result: filteredResults,
    nonce,
    serverSeed,
    clientSeed
  };
}
export { setData, ranges};
