const ranges = [
    { minRange: 1, maxRange: 56, tag: "T , Foil" },
    { minRange: 1, maxRange: 81, tag: "C , Foil" },
    { minRange: 1, maxRange: 81, tag: "C , Foil" },
    { minRange: 1, maxRange: 81, tag: "C , Foil" },
    { minRange: 1, maxRange: 81, tag: "C , Foil" },
    { minRange: 1, maxRange: 100, tag: "U , Foil" },
    { minRange: 1, maxRange: 100, tag: "U , Foil" },
    { minRange: 1, maxRange: 100, tag: "U , Foil" },
    { minRange: 1, maxRange: 15, tag: "L , Foil" },
    { minRange: 1, maxRange: 20, tag: "OTP U Show , Normal" },
    { minRange: 1, maxRange: 20, tag: "OTP U Show , Foil" },
    { minRange: 1, maxRange: 100, tag: "Slot11" },
    { minRange: 1, maxRange: 60, tag: "R , Foil" },
    { minRange: 1, maxRange: 20, tag: "M , Foil" },
    { minRange: 1, maxRange: 30, tag: "BIG M , Foil" },
    { minRange: 1, maxRange: 100, tag: "Slot12" },
    { minRange: 1, maxRange: 5, tag: "R Wanted , Normal" },
    { minRange: 1, maxRange: 8, tag: "M Wanted , Normal" },
    { minRange: 1, maxRange: 5, tag: "R Borderless , Normal" },
    { minRange: 1, maxRange: 2, tag: "M Borderless , Normal" },
    { minRange: 1, maxRange: 61, tag: "Extended Art , Normal" },
    { minRange: 1, maxRange: 30, tag: "BIG Extended Art , Normal" },
    { minRange: 1, maxRange: 30, tag: "BIG Vault , Normal" },
    { minRange: 1, maxRange: 100, tag: "Slot13" },
    { minRange: 1, maxRange: 36, tag: "OTC Extended Art , Normal" },
    { minRange: 1, maxRange: 4, tag: "OTC Commander , Normal" },
    { minRange: 1, maxRange: 4, tag: "OTC Commander , Foil" },
    { minRange: 1, maxRange: 100, tag: "Slot14" },
    { minRange: 1, maxRange: 30, tag: "OTP R Show , Normal" },
    { minRange: 1, maxRange: 15, tag: "OTP M Show , Normal" },
    { minRange: 1, maxRange: 200, tag: "Slot15" },
    { minRange: 1, maxRange: 61, tag: "Extended Art , Foil" },
    { minRange: 1, maxRange: 30, tag: "OTP R Show , Foil" },
    { minRange: 1, maxRange: 15, tag: "OTP M Show , Foil" },
    { minRange: 1, maxRange: 5, tag: "R Wanted , Foil" },
    { minRange: 1, maxRange: 8, tag: "M Wanted , Foil" },
    { minRange: 1, maxRange: 5, tag: "R Borderless , Foil" },
    { minRange: 1, maxRange: 2, tag: "M Borderless , Foil" },
    { minRange: 1, maxRange: 30, tag: "BIG Extended Art , Foil" },
    { minRange: 1, maxRange: 30, tag: "BIG Vault , Foil" },
    { minRange: 1, maxRange: 15, tag: "OTP Texture , Foil" },
    { minRange: 1, maxRange: 5, tag: "BIG Raised , Foil" },
    { minRange: 1, maxRange: 10, tag: "SPG , Foil" },
];

function incrementNonce(nonce, offset) {
  let seconds = parseInt(nonce.slice(0, 2), 10);
  let minutes = parseInt(nonce.slice(2, 4), 10);
  let hours = parseInt(nonce.slice(4, 6), 10);
  let day = parseInt(nonce.slice(6, 8), 10);
  let month = parseInt(nonce.slice(8, 10), 10);
  let year = parseInt(nonce.slice(10, 14), 10);

  seconds += offset;
  while (seconds > 59) {
    seconds -= 60;
    minutes += 1;
    if (minutes > 59) {
      minutes = 0;
      hours += 1;
      if (hours > 23) {
        hours = 0;
        day += 1;
        const daysInMonth = [31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31];
        if (day > daysInMonth[month - 1]) {
          day = 1;
          month += 1;
          if (month > 12) {
            month = 1;
            year += 1;
          }
        }
      }
    }
  }

  return (
    seconds.toString().padStart(2, '0') +
    minutes.toString().padStart(2, '0') +
    hours.toString().padStart(2, '0') +
    day.toString().padStart(2, '0') +
    month.toString().padStart(2, '0') +
    year.toString().padStart(4, '0')
  );
}

async function generateHmacSha512(key, message) {
  const keyBuffer = new TextEncoder().encode(key);
  const msgBuffer = new TextEncoder().encode(message);
  const cryptoKey = await crypto.subtle.importKey(
    "raw",
    keyBuffer,
    { name: "HMAC", hash: "SHA-512" },
    false,
    ["sign"]
  );
  const signature = await crypto.subtle.sign("HMAC", cryptoKey, msgBuffer);
  return new Uint8Array(signature);
}

async function generateRandomNumbers(ranges, serverSeed, clientSeed, nonce) {
  const randomNumbers = [];
  const numbersPerDigest = 16;
  const digestsNeeded = Math.ceil(ranges.length / numbersPerDigest);

  for (let digestIndex = 0; digestIndex < digestsNeeded; digestIndex++) {
    const message = `${clientSeed}:${nonce}:${digestIndex}`;
    const digest = await generateHmacSha512(serverSeed, message);
    let offset = 0;

    for (let i = 0; i < numbersPerDigest && randomNumbers.length < ranges.length; i++) {
      const range = ranges[randomNumbers.length];
      const bytes = digest.slice(offset, offset + 4);
      const view = new DataView(bytes.buffer, bytes.byteOffset, 4);
      const num = view.getUint32(0, false); // Big-endian
      const rand = Math.floor((num / 0x100000000) * (range.maxRange - range.minRange + 1)) + range.minRange;
      randomNumbers.push(rand);
      offset += 4;
    }
  }

  return randomNumbers;
}

async function mapTaggedResults(randomNumbers, ranges, serverSeed, clientSeed, nonce) {
  const seenValues = {};
  const taggedResults = [];

  for (let index = 0; index < randomNumbers.length; index++) {
    if (!ranges[index]) {
      console.error(`No range found for index ${index}`);
      continue;
    }

    const { tag = "UNKNOWN", minRange, maxRange } = ranges[index];
    let finalNumber = randomNumbers[index];
    let nonceOffset = 0;

    if (!seenValues[tag]) {
      seenValues[tag] = new Set();
    }

    while (seenValues[tag].has(finalNumber.toString())) {
      nonceOffset++;
      const newNonce = incrementNonce(nonce, nonceOffset);
      const message = `${clientSeed}:${newNonce}`;
      const digest = await generateHmacSha512(serverSeed, message);
      const bytes = digest.slice(0, 4);
      const view = new DataView(bytes.buffer, bytes.byteOffset, 4);
      const newNum = view.getUint32(0, false); // Big-endian
      finalNumber = Math.floor((newNum / 0x100000000) * (maxRange - minRange + 1)) + minRange;
    }

    seenValues[tag].add(finalNumber.toString());
    taggedResults.push({ value: finalNumber.toString(), tag });
  }

  return taggedResults;
}

async function filterResults(taggedResults, ranges, serverSeed, clientSeed, nonce) {
  let filteredResults = [];
  let nonceOffset = 0;
  const usedIndexes = new Set();

  const findAndConsume = (tag) => {
    const item = taggedResults.find((item, i) => item.tag === tag && !usedIndexes.has(i));
    if (item) {
      usedIndexes.add(taggedResults.indexOf(item));
      return item;
    }
    return null;
  };

  const add = (item) => { if (item) filteredResults.push(item); };

  let tFrameFoil = taggedResults.filter((item, i) => item.tag === "T , Foil" && !usedIndexes.has(i));
  tFrameFoil.slice(0, 1).forEach(item => {
    usedIndexes.add(taggedResults.indexOf(item));
    filteredResults.push(item);
  });

  let cFoil = taggedResults.filter((item, i) => item.tag === "C , Foil" && !usedIndexes.has(i));
  cFoil.slice(0, 4).forEach(item => {
    usedIndexes.add(taggedResults.indexOf(item));
    filteredResults.push(item);
  });

  let uFoil = taggedResults.filter((item, i) => item.tag === "U , Foil" && !usedIndexes.has(i));
  uFoil.slice(0, 3).forEach(item => {
    usedIndexes.add(taggedResults.indexOf(item));
    filteredResults.push(item);
  });

  let lFoil = taggedResults.filter((item, i) => item.tag === "L , Foil" && !usedIndexes.has(i));
  lFoil.slice(0, 1).forEach(item => {
    usedIndexes.add(taggedResults.indexOf(item));
    filteredResults.push(item);
  });

  let usNorm = taggedResults.filter((item, i) => item.tag === "OTP U Show , Normal" && !usedIndexes.has(i));
  usNorm.slice(0, 1).forEach(item => {
    usedIndexes.add(taggedResults.indexOf(item));
    filteredResults.push(item);
  });

  let usFoil = taggedResults.filter((item, i) => item.tag === "OTP U Show , Foil" && !usedIndexes.has(i));
  usFoil.slice(0, 1).forEach(item => {
    usedIndexes.add(taggedResults.indexOf(item));
    filteredResults.push(item);
  });

  const slot11 = taggedResults.find(item => item.tag === "Slot11");
  let slot11Item = null;
  let slot11Tag = null;
  if (slot11) {
    const num = parseInt(slot11.value, 10);
    if (num <= 70) {
      slot11Tag = "R , Foil";
      slot11Item = findAndConsume("R , Foil");
    } else if (num <= 82) {
      slot11Tag = "M , Foil";
      slot11Item = findAndConsume("M , Foil");
    } else if (num <= 100) {
      slot11Tag = "BIG M , Foil";
      slot11Item = findAndConsume("BIG M , Foil");
    }
    add(slot11Item);
  }

  const slot12 = taggedResults.find(item => item.tag === "Slot12");
  let slot12Item = null;
  let slot12Tag = null;
  if (slot12) {
    const num = parseInt(slot12.value, 10);
    if (num <= 4) {
      slot12Tag = "R Wanted , Normal";
      slot12Item = findAndConsume("R Wanted , Normal");
    } else if (num <= 10) {
      slot12Tag = "M Wanted , Normal";
      slot12Item = findAndConsume("M Wanted , Normal");
    } else if (num <= 14) {
      slot12Tag = "R Borderless , Normal";
      slot12Item = findAndConsume("R Borderless , Normal");
    } else if (num <= 15) {
      slot12Tag = "M Borderless , Normal";
      slot12Item = findAndConsume("M Borderless , Normal");
    } else if (num <= 58) {
      slot12Tag = "Extended Art , Normal";
      slot12Item = findAndConsume("Extended Art , Normal");
    } else if (num <= 79) {
      slot12Tag = "BIG Extended Art , Normal";
      slot12Item = findAndConsume("BIG Extended Art , Normal");
    } else if (num <= 100) {
      slot12Tag = "BIG Vault , Normal";
      slot12Item = findAndConsume("BIG Vault , Normal");
    }
    add(slot12Item);
  }

  if (slot11Item && slot12Item && slot11Tag === slot12Tag && slot11Item.value === slot12Item.value) {
    const rangeForTag = ranges.find(range => range.tag === slot12Tag);
    if (rangeForTag) {
      const { minRange, maxRange } = rangeForTag;
      let newValue;
      do {
        nonceOffset++;
        const newNonce = incrementNonce(nonce, nonceOffset);
        const message = `${clientSeed}:${newNonce}`;
        const digest = await generateHmacSha512(serverSeed, message);
        const bytes = digest.slice(0, 4);
        const view = new DataView(bytes.buffer, bytes.byteOffset, 4);
        const newNum = view.getUint32(0, false); // Big-endian
        newValue = Math.floor((newNum / 0x100000000) * (maxRange - minRange + 1)) + minRange;
      } while (newValue.toString() === slot11Item.value);
      slot12Item = { value: newValue.toString(), tag: slot12Tag };
      add(slot12Item);
    }
  }

  const slot13 = taggedResults.find(item => item.tag === "Slot13");
  if (slot13) {
    const num = parseInt(slot13.value, 10);
    if (num <= 82) {
      add(findAndConsume("OTC Extended Art , Normal"));
    } else if (num <= 94) {
      add(findAndConsume("OTC Commander , Normal"));
    } else if (num <= 100) {
      add(findAndConsume("OTC Commander , Foil"));
    }
  }

  const slot14 = taggedResults.find(item => item.tag === "Slot14");
  if (slot14) {
    const num = parseInt(slot14.value, 10);
    if (num <= 80) {
      add(findAndConsume("OTP R Show , Normal"));
    } else if (num <= 100) {
      add(findAndConsume("OTP M Show , Normal"));
    }
  }

  const slot15 = taggedResults.find(item => item.tag === "Slot15");
  if (slot15) {
    const num = parseInt(slot15.value, 10);
    if (num <= 91) {
      add(findAndConsume("Extended Art , Foil"));
    } else if (num <= 131) {
      add(findAndConsume("OTP R Show , Foil"));
    } else if (num <= 151) {
      add(findAndConsume("OTP M Show , Foil"));
    } else if (num <= 159) {
      add(findAndConsume("R Wanted , Foil"));
    } else if (num <= 163) {
      add(findAndConsume("M Wanted , Foil"));
    } else if (num <= 169) {
      add(findAndConsume("R Borderless , Foil"));
    } else if (num <= 171) {
      add(findAndConsume("M Borderless , Foil"));
    } else if (num <= 181) {
      add(findAndConsume("BIG Extended Art , Foil"));
    } else if (num <= 191) {
      add(findAndConsume("BIG Vault , Foil"));
    } else if (num <= 193) {
      add(findAndConsume("OTP Texture , Foil"));
    } else if (num <= 194) {
      add(findAndConsume("BIG Raised , Foil"));
    } else if (num <= 200) {
      add(findAndConsume("SPG , Foil"));
    }
  }

  return filteredResults;
}

async function setData(serverSeed, clientSeed, nonce) {
  const randomNumbers = await generateRandomNumbers(ranges, serverSeed, clientSeed, nonce);
  const taggedResults = await mapTaggedResults(randomNumbers, ranges, serverSeed, clientSeed, nonce);
  const filteredResults = await filterResults(taggedResults, ranges, serverSeed, clientSeed, nonce);

  return {
    success: true,
    result: filteredResults,
    nonce,
    serverSeed,
    clientSeed
  };
}
export { setData, ranges};
