"use strict";
const crypto = require('crypto');

const ranges = [
    { minRange: 1, maxRange: 2, tag: "Code Card" },
    { minRange: 1, maxRange: 8, tag: "Energy" },
    { minRange: 1, maxRange: 89, tag: "Common" },
    { minRange: 1, maxRange: 89, tag: "Common" },
    { minRange: 1, maxRange: 89, tag: "Common" },
    { minRange: 1, maxRange: 89, tag: "Common" },
    { minRange: 1, maxRange: 89, tag: "Common" },
    { minRange: 1, maxRange: 83, tag: "Uncommon" },
    { minRange: 1, maxRange: 83, tag: "Uncommon" },
    { minRange: 1, maxRange: 83, tag: "Uncommon" },
    { minRange: 1, maxRange: 217, tag: "Reverse Holofoil" },
    { minRange: 1, maxRange: 10000, tag: "slot10" },
    { minRange: 1, maxRange: 32, tag: "Rare" },
    { minRange: 1, maxRange: 13, tag: "Holo Rare" },
    { minRange: 1, maxRange: 8, tag: "Holo Vmax" },
    { minRange: 1, maxRange: 29, tag: "Full Art V" },
    { minRange: 1, maxRange: 6, tag: "Full Art Trainer" },
    { minRange: 1, maxRange: 10, tag: "Rainbow Rare" },
    { minRange: 1, maxRange: 5, tag: "Golden Rare" },
    { minRange: 1, maxRange: 4, tag: "Alt Art V" },
    { minRange: 1, maxRange: 5, tag: "Alt Art Vmax" },
];

function incrementNonce(nonce, offset) {
    let seconds = parseInt(nonce.slice(0, 2), 10);
    let minutes = parseInt(nonce.slice(2, 4), 10);
    let hours = parseInt(nonce.slice(4, 6), 10);
    let day = parseInt(nonce.slice(6, 8), 10);
    let month = parseInt(nonce.slice(8, 10), 10);
    let year = parseInt(nonce.slice(10, 14), 10);
    seconds += offset;
    while (seconds > 59) {
        seconds -= 60;
        minutes += 1;
        if (minutes > 59) {
            minutes = 0;
            hours += 1;
            if (hours > 23) {
                hours = 0;
                day += 1;
                const daysInMonth = [31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31];
                if (day > daysInMonth[month - 1]) {
                    day = 1;
                    month += 1;
                    if (month > 12) {
                        month = 1;
                        year += 1;
                    }
                }
            }
        }
    }
    return (
        seconds.toString().padStart(2, '0') +
        minutes.toString().padStart(2, '0') +
        hours.toString().padStart(2, '0') +
        day.toString().padStart(2, '0') +
        month.toString().padStart(2, '0') +
        year.toString().padStart(4, '0')
    );
}

function generateHmacSha512(key, message) {
    const hmac = crypto.createHmac('sha512', key);
    hmac.update(message);
    return hmac.digest();
}

function generateRandomNumbers(ranges, serverSeed, clientSeed, nonce) {
    const randomNumbers = [];
    const numbersPerDigest = 16;
    const digestsNeeded = Math.ceil(ranges.length / numbersPerDigest);
    for (let digestIndex = 0; digestIndex < digestsNeeded; digestIndex++) {
        const hmac = crypto.createHmac('sha512', serverSeed);
        hmac.update(`${clientSeed}:${nonce}:${digestIndex}`);
        const digest = hmac.digest();
        let offset = 0;
        for (let i = 0; i < numbersPerDigest && randomNumbers.length < ranges.length; i++) {
            const range = ranges[randomNumbers.length];
            const bytes = digest.slice(offset, offset + 4);
            const num = bytes.readUInt32BE(0);
            const rand = Math.floor((num / 0x100000000) * (range.maxRange - range.minRange + 1)) + range.minRange;
            randomNumbers.push(rand);
            offset += 4;
        }
    }
    return randomNumbers;
}

function mapTaggedResults(randomNumbers, ranges, serverSeed, clientSeed, nonce) {
    const seenValues = {};
    const taggedResults = [];
    randomNumbers.forEach((num, index) => {
        if (!ranges[index]) {
            console.error(`No range found for index ${index}`);
            return;
        }
        const { tag = "UNKNOWN", minRange, maxRange } = ranges[index];
        let finalNumber = num;
        let nonceOffset = 0;
        if (!seenValues[tag]) {
            seenValues[tag] = new Set();
        }
        while (seenValues[tag].has(finalNumber.toString())) {
            nonceOffset++;
            const newNonce = incrementNonce(nonce, nonceOffset);
            const hmac = crypto.createHmac('sha512', serverSeed);
            hmac.update(`${clientSeed}:${newNonce}`);
            const digest = hmac.digest();
            const bytes = digest.slice(0, 4);
            const newNum = bytes.readUInt32BE(0);
            finalNumber = Math.floor((newNum / 0x100000000) * (maxRange - minRange + 1)) + minRange;
        }
        seenValues[tag].add(finalNumber.toString());
        taggedResults.push({ value: finalNumber.toString(), tag });
    });
    return taggedResults;
}

function filterResults(taggedResults, ranges) {
    let filteredResults = [];
    const usedValuesByTag = {};

    function getNextCard(tag) {
        if (!usedValuesByTag[tag]) {
            usedValuesByTag[tag] = new Set();
        }
        const cards = taggedResults.filter(item => item.tag === tag);
        for (const card of cards) {
            if (!usedValuesByTag[tag].has(card.value)) {
                usedValuesByTag[tag].add(card.value);
                return card;
            }
        }
        return null;
    }

    const codeCard = taggedResults.find((item) => item.tag === "Code Card");
    if (codeCard) filteredResults.push(codeCard);

    const energyCard = taggedResults.find((item) => item.tag === "Energy");
    if (energyCard) filteredResults.push(energyCard);

    const commonCards = taggedResults.filter((item) => item.tag === "Common");
    filteredResults.push(...commonCards.slice(0, 5));

    const uncommonCards = taggedResults.filter((item) => item.tag === "Uncommon");
    filteredResults.push(...uncommonCards.slice(0, 3));

    const reverseHolo = getNextCard("Reverse Holofoil");
    if (reverseHolo) filteredResults.push(reverseHolo);

    const rareCategoryItem = taggedResults.find((item) => item.tag === "slot10");
    if (rareCategoryItem) {
        const rareCategoryNum = parseInt(rareCategoryItem.value, 10);
        if (rareCategoryNum >= 1 && rareCategoryNum <= 6061) {
            const rareCard = getNextCard("Rare");
            if (rareCard) filteredResults.push(rareCard);
        } else if (rareCategoryNum >= 6062 && rareCategoryNum <= 9091) {
            const rareCard = getNextCard("Holo Rare");
            if (rareCard) filteredResults.push(rareCard);
        } else if (rareCategoryNum >= 9092 && rareCategoryNum <= 9424) {
            const rareCard = getNextCard("Holo Vmax");
            if (rareCard) filteredResults.push(rareCard);
        } else if (rareCategoryNum >= 9425 && rareCategoryNum <= 9596) {
            const rareCard = getNextCard("Full Art V");
            if (rareCard) filteredResults.push(rareCard);
        } else if (rareCategoryNum >= 9597 && rareCategoryNum <= 9752) {
            const rareCard = getNextCard("Full Art Trainer");
            if (rareCard) filteredResults.push(rareCard);
        } else if (rareCategoryNum >= 9753 && rareCategoryNum <= 9831) {
            const rareCard = getNextCard("Rainbow Rare");
            if (rareCard) filteredResults.push(rareCard);
        } else if (rareCategoryNum >= 9832 && rareCategoryNum <= 9914) {
            const rareCard = getNextCard("Golden Rare");
            if (rareCard) filteredResults.push(rareCard);
        } else if (rareCategoryNum >= 9915 && rareCategoryNum <= 9970) {
            const rareCard = getNextCard("Alt Art V");
            if (rareCard) filteredResults.push(rareCard);
        } else if (rareCategoryNum >= 9971 && rareCategoryNum <= 10000) {
            const rareCard = getNextCard("Alt Art Vmax");
            if (rareCard) filteredResults.push(rareCard);
        }
    }

    return filteredResults;
}

async function setData(serverSeed, clientSeed, nonce) {
    const randomNumbers = generateRandomNumbers(ranges, serverSeed, clientSeed, nonce);
    const taggedResults = mapTaggedResults(randomNumbers, ranges, serverSeed, clientSeed, nonce);
    const filteredResults = filterResults(taggedResults, ranges);
    return {
        success: true,
        result: filteredResults,
        nonce,
        serverSeed,
        clientSeed
    };
}

export { setData, ranges };