const ranges = [
    { minRange: 1, maxRange: 31, tag: "T , Foil" },
    { minRange: 1, maxRange: 81, tag: "C , Foil" },
    { minRange: 1, maxRange: 81, tag: "C , Foil" },
    { minRange: 1, maxRange: 81, tag: "C , Foil" },
    { minRange: 1, maxRange: 81, tag: "C , Foil" },
    { minRange: 1, maxRange: 81, tag: "C , Foil" },
    { minRange: 1, maxRange: 100, tag: "U , Foil" },
    { minRange: 1, maxRange: 100, tag: "U , Foil" },
    { minRange: 1, maxRange: 100, tag: "U , Foil" },
    { minRange: 1, maxRange: 100, tag: "U , Foil" },
    { minRange: 1, maxRange: 5, tag: "L , Foil" },
    { minRange: 1, maxRange: 100, tag: "Slot11" },
    { minRange: 1, maxRange: 60, tag: "R , Foil" },
    { minRange: 1, maxRange: 20, tag: "M , Foil" },
    { minRange: 1, maxRange: 100, tag: "Slot12" },
    { minRange: 1, maxRange: 27, tag: "DSC Extended Art , Normal" },
    { minRange: 1, maxRange: 8, tag: "DSC Commander , Normal" },
    { minRange: 1, maxRange: 8, tag: "DSC Commander , Foil" },
    { minRange: 1, maxRange: 100, tag: "Slot13" },
    { minRange: 1, maxRange: 18, tag: "Extended Art , Normal" },
    { minRange: 1, maxRange: 26, tag: "Showcase , Normal" },
    { minRange: 1, maxRange: 14, tag: "R Borderless , Normal" },
    { minRange: 1, maxRange: 9, tag: "M Borderless , Normal" },
    { minRange: 1, maxRange: 12, tag: "R Double , Normal" },
    { minRange: 1, maxRange: 5, tag: "M Double , Normal" },
    { minRange: 1, maxRange: 100, tag: "Slot14" },
    { minRange: 1, maxRange: 18, tag: "Extended Art , Normal" },
    { minRange: 1, maxRange: 26, tag: "Showcase , Normal" },
    { minRange: 1, maxRange: 14, tag: "R Borderless , Normal" },
    { minRange: 1, maxRange: 9, tag: "M Borderless , Normal" },
    { minRange: 1, maxRange: 12, tag: "R Double , Normal" },
    { minRange: 1, maxRange: 5, tag: "M Double , Normal" },
    { minRange: 1, maxRange: 100, tag: "Slot15" },
    { minRange: 1, maxRange: 18, tag: "Extended Art , Foil" },
    { minRange: 1, maxRange: 26, tag: "Showcase , Foil" },
    { minRange: 1, maxRange: 14, tag: "R Borderless , Foil" },
    { minRange: 1, maxRange: 9, tag: "M Borderless , Foil" },
    { minRange: 1, maxRange: 5, tag: "R Double , Foil" },
    { minRange: 1, maxRange: 5, tag: "M Double , Foil" },
    { minRange: 1, maxRange: 5, tag: "Textured , Foil" },
    { minRange: 1, maxRange: 10, tag: "Japan , Foil" },
    { minRange: 1, maxRange: 10, tag: "Fracture , Foil" },
    { minRange: 1, maxRange: 20, tag: "SPG , Foil" },
];

function incrementNonce(nonce, offset) {
  let seconds = parseInt(nonce.slice(0, 2), 10);
  let minutes = parseInt(nonce.slice(2, 4), 10);
  let hours = parseInt(nonce.slice(4, 6), 10);
  let day = parseInt(nonce.slice(6, 8), 10);
  let month = parseInt(nonce.slice(8, 10), 10);
  let year = parseInt(nonce.slice(10, 14), 10);

  seconds += offset;
  while (seconds > 59) {
    seconds -= 60;
    minutes += 1;
    if (minutes > 59) {
      minutes = 0;
      hours += 1;
      if (hours > 23) {
        hours = 0;
        day += 1;
        const daysInMonth = [31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31];
        if (day > daysInMonth[month - 1]) {
          day = 1;
          month += 1;
          if (month > 12) {
            month = 1;
            year += 1;
          }
        }
      }
    }
  }

  return (
    seconds.toString().padStart(2, '0') +
    minutes.toString().padStart(2, '0') +
    hours.toString().padStart(2, '0') +
    day.toString().padStart(2, '0') +
    month.toString().padStart(2, '0') +
    year.toString().padStart(4, '0')
  );
}

async function generateHmacSha512(key, message) {
  const keyBuffer = new TextEncoder().encode(key);
  const msgBuffer = new TextEncoder().encode(message);
  const cryptoKey = await crypto.subtle.importKey(
    "raw",
    keyBuffer,
    { name: "HMAC", hash: "SHA-512" },
    false,
    ["sign"]
  );
  const signature = await crypto.subtle.sign("HMAC", cryptoKey, msgBuffer);
  return new Uint8Array(signature);
}

async function generateRandomNumbers(ranges, serverSeed, clientSeed, nonce) {
  const randomNumbers = [];
  const numbersPerDigest = 16;
  const digestsNeeded = Math.ceil(ranges.length / numbersPerDigest);

  for (let digestIndex = 0; digestIndex < digestsNeeded; digestIndex++) {
    const message = `${clientSeed}:${nonce}:${digestIndex}`;
    const digest = await generateHmacSha512(serverSeed, message);
    let offset = 0;

    for (let i = 0; i < numbersPerDigest && randomNumbers.length < ranges.length; i++) {
      const range = ranges[randomNumbers.length];
      const bytes = digest.slice(offset, offset + 4);
      const view = new DataView(bytes.buffer, bytes.byteOffset, 4);
      const num = view.getUint32(0, false); // Big-endian
      const rand = Math.floor((num / 0x100000000) * (range.maxRange - range.minRange + 1)) + range.minRange;
      randomNumbers.push(rand);
      offset += 4;
    }
  }

  return randomNumbers;
}

async function mapTaggedResults(randomNumbers, ranges, serverSeed, clientSeed, nonce) {
  const seenValues = {};
  const taggedResults = [];

  for (let index = 0; index < randomNumbers.length; index++) {
    if (!ranges[index]) {
      console.error(`No range found for index ${index}`);
      continue;
    }

    const { tag = "UNKNOWN", minRange, maxRange } = ranges[index];
    let finalNumber = randomNumbers[index];
    let nonceOffset = 0;

    if (!seenValues[tag]) {
      seenValues[tag] = new Set();
    }

    while (seenValues[tag].has(finalNumber.toString())) {
      nonceOffset++;
      const newNonce = incrementNonce(nonce, nonceOffset);
      const message = `${clientSeed}:${newNonce}`;
      const digest = await generateHmacSha512(serverSeed, message);
      const bytes = digest.slice(0, 4);
      const view = new DataView(bytes.buffer, bytes.byteOffset, 4);
      const newNum = view.getUint32(0, false); // Big-endian
      finalNumber = Math.floor((newNum / 0x100000000) * (maxRange - minRange + 1)) + minRange;
    }

    seenValues[tag].add(finalNumber.toString());
    taggedResults.push({ value: finalNumber.toString(), tag });
  }

  return taggedResults;
}

async function filterResults(taggedResults, ranges, serverSeed, clientSeed, nonce) {
  let filteredResults = [];
  const usedIndexes = new Set();
  let nonceOffset = 0;

  const findAndConsume = (tag) => {
    const item = taggedResults.find((item, i) => item.tag === tag && !usedIndexes.has(i));
    if (item) usedIndexes.add(taggedResults.indexOf(item));
    return item;
  };

  const add = (item) => { if (item) filteredResults.push(item); };

  let tFrameFoil = taggedResults.filter((item, i) => item.tag === "T , Foil" && !usedIndexes.has(i));
  tFrameFoil.slice(0, 1).forEach(item => {
    usedIndexes.add(taggedResults.indexOf(item));
    filteredResults.push(item);
  });

  let cFoil = taggedResults.filter((item, i) => item.tag === "C , Foil" && !usedIndexes.has(i));
  cFoil.slice(0, 5).forEach(item => {
    usedIndexes.add(taggedResults.indexOf(item));
    filteredResults.push(item);
  });

  let uFoil = taggedResults.filter((item, i) => item.tag === "U , Foil" && !usedIndexes.has(i));
  uFoil.slice(0, 4).forEach(item => {
    usedIndexes.add(taggedResults.indexOf(item));
    filteredResults.push(item);
  });

  let lFoil = taggedResults.filter((item, i) => item.tag === "L , Foil" && !usedIndexes.has(i));
  lFoil.slice(0, 1).forEach(item => {
    usedIndexes.add(taggedResults.indexOf(item));
    filteredResults.push(item);
  });

  const slot11 = taggedResults.find(item => item.tag === "Slot11");
  let slot11Item = null;
  let slot11Tag = null;
  if (slot11) {
    const num = parseInt(slot11.value, 10);
    if (num <= 87) {
      slot11Tag = "R , Foil";
      slot11Item = findAndConsume("R , Foil");
    } else {
      slot11Tag = "M , Foil";
      slot11Item = findAndConsume("M , Foil");
    }
    add(slot11Item);
  }

  const slot12 = taggedResults.find(item => item.tag === "Slot12");
  let slot12Item = null;
  let slot12Tag = null;
  if (slot12) {
    const num = parseInt(slot12.value, 10);
    if (num <= 84) {
      slot12Tag = "DSC Extended Art , Normal";
      slot12Item = findAndConsume("DSC Extended Art , Normal");
    } else if (num <= 95) {
      slot12Tag = "DSC Commander , Normal";
      slot12Item = findAndConsume("DSC Commander , Normal");
    } else {
      slot12Tag = "DSC Commander , Foil";
      slot12Item = findAndConsume("DSC Commander , Foil");
    }
    add(slot12Item);
  }

  const slot13 = taggedResults.find(item => item.tag === "Slot13");
  let slot13Item = null;
  let slot13Tag = null;
  if (slot13) {
    const num = parseInt(slot13.value, 10);
    if (num <= 21) {
      slot13Tag = "Extended Art , Normal";
      slot13Item = findAndConsume("Extended Art , Normal");
    } else if (num <= 52) {
      slot13Tag = "Showcase , Normal";
      slot13Item = findAndConsume("Showcase , Normal");
    } else if (num <= 69) {
      slot13Tag = "R Borderless , Normal";
      slot13Item = findAndConsume("R Borderless , Normal");
    } else if (num <= 80) {
      slot13Tag = "M Borderless , Normal";
      slot13Item = findAndConsume("M Borderless , Normal");
    } else if (num <= 94) {
      slot13Tag = "R Double , Normal";
      slot13Item = findAndConsume("R Double , Normal");
    } else {
      slot13Tag = "M Double , Normal";
      slot13Item = findAndConsume("M Double , Normal");
    }
    add(slot13Item);
  }

  const slot14 = taggedResults.find(item => item.tag === "Slot14");
  let slot14Item = null;
  let slot14Tag = null;
  if (slot14) {
    const num = parseInt(slot14.value, 10);
    if (num <= 21) {
      slot14Tag = "Extended Art , Normal";
      slot14Item = findAndConsume("Extended Art , Normal");
    } else if (num <= 52) {
      slot14Tag = "Showcase , Normal";
      slot14Item = findAndConsume("Showcase , Normal");
    } else if (num <= 69) {
      slot14Tag = "R Borderless , Normal";
      slot14Item = findAndConsume("R Borderless , Normal");
    } else if (num <= 80) {
      slot14Tag = "M Borderless , Normal";
      slot14Item = findAndConsume("M Borderless , Normal");
    } else if (num <= 94) {
      slot14Tag = "R Double , Normal";
      slot14Item = findAndConsume("R Double , Normal");
    } else {
      slot14Tag = "M Double , Normal";
      slot14Item = findAndConsume("M Double , Normal");
    }

    if (slot13Item && slot14Item && slot13Tag === slot14Tag && slot13Item.value === slot14Item.value) {
      const rangeForTag = ranges.find(range => range.tag === slot14Tag);
      if (rangeForTag) {
        const { minRange, maxRange } = rangeForTag;
        let newValue;
        do {
          nonceOffset++;
          const newNonce = incrementNonce(nonce, nonceOffset);
          const message = `${clientSeed}:${newNonce}`;
          const digest = await generateHmacSha512(serverSeed, message);
          const bytes = digest.slice(0, 4);
          const view = new DataView(bytes.buffer, bytes.byteOffset, 4);
          const newNum = view.getUint32(0, false); // Big-endian
          newValue = Math.floor((newNum / 0x100000000) * (maxRange - minRange + 1)) + minRange;
        } while (newValue.toString() === slot13Item.value);

        slot14Item = { value: newValue.toString(), tag: slot14Tag };
      }
    }
    add(slot14Item);
  }

  const slot15 = taggedResults.find(item => item.tag === "Slot15");
  if (slot15) {
    const num = parseInt(slot15.value, 10);
    if (num <= 21) {
      add(findAndConsume("Extended Art , Foil"));
    } else if (num <= 49) {
      add(findAndConsume("Showcase , Foil"));
    } else if (num <= 62) {
      add(findAndConsume("R Borderless , Foil"));
    } else if (num <= 66) {
      add(findAndConsume("M Borderless , Foil"));
    } else if (num <= 82) {
      add(findAndConsume("R Double , Foil"));
    } else if (num <= 85) {
      add(findAndConsume("M Double , Foil"));
    } else if (num <= 86) {
      add(findAndConsume("Textured , Foil"));
    } else if (num <= 96) {
      add(findAndConsume("Japan , Foil"));
    } else if (num <= 97) {
      add(findAndConsume("Fracture , Foil"));
    } else {
      add(findAndConsume("SPG , Foil"));
    }
  }

  return filteredResults;
}

async function setData(serverSeed, clientSeed, nonce) {
  const randomNumbers = await generateRandomNumbers(ranges, serverSeed, clientSeed, nonce);
  const taggedResults = await mapTaggedResults(randomNumbers, ranges, serverSeed, clientSeed, nonce);
  const filteredResults = await filterResults(taggedResults, ranges, serverSeed, clientSeed, nonce);

  return {
    success: true,
    result: filteredResults,
    nonce,
    serverSeed,
    clientSeed
  };
}
export { setData, ranges};
